<?php
// admin-handler.php - Handles admin panel requests

// Set JSON response header
header('Content-Type: application/json');

// Start session for admin authentication
if (session_status() === PHP_SESSION_NONE) {
    // Use httponly cookie; secure flag left default so it works on HTTP development servers
    session_start();
}

// Log the request for debugging
error_log("Admin request received. Method: " . $_SERVER['REQUEST_METHOD']);

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Check for file upload first (multipart/form-data has no JSON body)
if (!empty($_FILES)) {
    // Helper to sanitize a provided base name into a safe slug
    function sanitize_base_name($name) {
        $name = mb_strtolower($name, 'UTF-8');
        $name = preg_replace('/\s*\([^\)]*\)\s*/u', ' ', $name);
        $name = preg_replace('/[^\p{L}\p{N}]+/u', '-', $name);
        if (function_exists('iconv')) {
            $name = iconv('UTF-8', 'ASCII//TRANSLIT//IGNORE', $name);
        }
        $name = strtolower($name);
        $name = preg_replace('/-+/', '-', $name);
        $name = trim($name, '-');
        return substr($name, 0, 120);
    }

    // Helper to update homepage HTML with hardcoded image URLs
    function updateHomepageImageUrls() {
        $imagesConfigFile = __DIR__ . '/../config/home-images.json';
        $indexFile = __DIR__ . '/../index.html';
        
        if (!file_exists($imagesConfigFile) || !file_exists($indexFile)) {
            return;
        }

        $imagesConfig = json_decode(file_get_contents($imagesConfigFile), true) ?: [];
        $htmlContent = file_get_contents($indexFile);

        // Update homeImage1 src
        if (!empty($imagesConfig['image1'])) {
            $htmlContent = preg_replace(
                '/id="homeImage1"\s+src="[^"]*"/i',
                'id="homeImage1" src="' . addslashes($imagesConfig['image1']) . '"',
                $htmlContent
            );
        }

        // Update homeImage2 src
        if (!empty($imagesConfig['image2'])) {
            $htmlContent = preg_replace(
                '/id="homeImage2"\s+src="[^"]*"/i',
                'id="homeImage2" src="' . addslashes($imagesConfig['image2']) . '"',
                $htmlContent
            );
        }

        file_put_contents($indexFile, $htmlContent);
    }

    // Check for homepage image upload
    if (isset($_FILES['homeImage']) && isset($_POST['position'])) {
        $file = $_FILES['homeImage'];
        $position = $_POST['position'] ?? null;

        if (!in_array($position, ['image1', 'image2'])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid position']);
            exit;
        }

        if ($file['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Upload error']);
            exit;
        }

        $maxBytes = 5 * 1024 * 1024;
        if ($file['size'] > $maxBytes) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'File too large']);
            exit;
        }

        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mime = $finfo->file($file['tmp_name']);
        $allowed = ['image/jpeg' => 'jpg', 'image/png' => 'png', 'image/webp' => 'webp'];
        if (!isset($allowed[$mime])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid file type']);
            exit;
        }

        $uploadDir = __DIR__ . '/../uploads/images';
        $publicPrefix = '/uploads/images/';

        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
            $ht = "Options -Indexes\n\n<FilesMatch \"\\.(php|php5|phtml|pl|py|jsp|asp|sh|cgi)$\">\n  Require all denied\n</FilesMatch>\n\n<IfModule mod_headers.c>\n  <FilesMatch \"\\.(jpg|jpeg|png|webp|gif)$\">\n    Header set Cache-Control \"max-age=604800, public\"\n  </FilesMatch>\n</IfModule>\n";
            @file_put_contents($uploadDir . '/.htaccess', $ht);
        }

        $ext = $allowed[$mime];
        $filename = 'homepage-' . $position . '-' . time() . '.' . $ext;
        $destination = $uploadDir . '/' . $filename;

        if (!move_uploaded_file($file['tmp_name'], $destination)) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Failed to save file']);
            exit;
        }

        // Load current images config
        $imagesConfigFile = __DIR__ . '/../config/home-images.json';
        $imagesConfig = [];
        if (file_exists($imagesConfigFile)) {
            $imagesConfig = json_decode(file_get_contents($imagesConfigFile), true) ?: [];
        }

        // Delete old image if it exists
        if (!empty($imagesConfig[$position])) {
            $oldPath = ltrim($imagesConfig[$position], '/');
            $oldFull = realpath(__DIR__ . '/../' . $oldPath);
            $uploadDirReal = realpath($uploadDir);
            if ($oldFull && $uploadDirReal && strpos($oldFull, $uploadDirReal) === 0 && is_file($oldFull)) {
                @unlink($oldFull);
            }
        }

        // Update config
        $imagesConfig[$position] = $publicPrefix . $filename;
        file_put_contents($imagesConfigFile, json_encode($imagesConfig, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        // Hardcode URLs into homepage HTML
        updateHomepageImageUrls();

        echo json_encode(['success' => true, 'url' => $publicPrefix . $filename]);
        exit;
    }

    // Check for favicon upload
    if (isset($_FILES['favicon'])) {
        $file = $_FILES['favicon'];
        
        if ($file['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Upload error']);
            exit;
        }

        $maxBytes = 1 * 1024 * 1024;
        if ($file['size'] > $maxBytes) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'File too large']);
            exit;
        }

        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mime = $finfo->file($file['tmp_name']);
        $allowed = ['image/x-icon' => 'ico', 'image/png' => 'png'];
        if (!isset($allowed[$mime])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid file type']);
            exit;
        }

        $uploadDir = __DIR__ . '/../uploads/images';
        $publicPrefix = '/uploads/images/';

        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
            $ht = "Options -Indexes\n\n<FilesMatch \"\\.(php|php5|phtml|pl|py|jsp|asp|sh|cgi)$\">\n  Require all denied\n</FilesMatch>\n\n<IfModule mod_headers.c>\n  <FilesMatch \"\\.(jpg|jpeg|png|webp|gif|ico)$\">\n    Header set Cache-Control \"max-age=604800, public\"\n  </FilesMatch>\n</IfModule>\n";
            @file_put_contents($uploadDir . '/.htaccess', $ht);
        }

        $ext = $allowed[$mime];
        $filename = 'favicon-' . time() . '.' . $ext;
        $destination = $uploadDir . '/' . $filename;

        if (!move_uploaded_file($file['tmp_name'], $destination)) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Failed to save file']);
            exit;
        }

        // Load current assets config
        $assetsConfigFile = __DIR__ . '/../config/site-assets.json';
        $assetsConfig = [];
        if (file_exists($assetsConfigFile)) {
            $assetsConfig = json_decode(file_get_contents($assetsConfigFile), true) ?: [];
        }

        // Delete old favicon if it exists
        if (!empty($assetsConfig['favicon'])) {
            $oldPath = ltrim($assetsConfig['favicon'], '/');
            $oldFull = realpath(__DIR__ . '/../' . $oldPath);
            $uploadDirReal = realpath($uploadDir);
            if ($oldFull && $uploadDirReal && strpos($oldFull, $uploadDirReal) === 0 && is_file($oldFull)) {
                @unlink($oldFull);
            }
        }

        // Update config
        $assetsConfig['favicon'] = $publicPrefix . $filename;
        file_put_contents($assetsConfigFile, json_encode($assetsConfig, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        echo json_encode(['success' => true, 'url' => $publicPrefix . $filename]);
        exit;
    }

    // Check for social logo upload
    if (isset($_FILES['socialLogo'])) {
        $file = $_FILES['socialLogo'];
        
        if ($file['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Upload error']);
            exit;
        }

        $maxBytes = 2 * 1024 * 1024;
        if ($file['size'] > $maxBytes) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'File too large']);
            exit;
        }

        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mime = $finfo->file($file['tmp_name']);
        $allowed = ['image/jpeg' => 'jpg', 'image/png' => 'png', 'image/webp' => 'webp'];
        if (!isset($allowed[$mime])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid file type']);
            exit;
        }

        $uploadDir = __DIR__ . '/../uploads/images';
        $publicPrefix = '/uploads/images/';

        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
            $ht = "Options -Indexes\n\n<FilesMatch \"\\.(php|php5|phtml|pl|py|jsp|asp|sh|cgi)$\">\n  Require all denied\n</FilesMatch>\n\n<IfModule mod_headers.c>\n  <FilesMatch \"\\.(jpg|jpeg|png|webp|gif|ico)$\">\n    Header set Cache-Control \"max-age=604800, public\"\n  </FilesMatch>\n</IfModule>\n";
            @file_put_contents($uploadDir . '/.htaccess', $ht);
        }

        $ext = $allowed[$mime];
        $filename = 'social-logo-' . time() . '.' . $ext;
        $destination = $uploadDir . '/' . $filename;

        if (!move_uploaded_file($file['tmp_name'], $destination)) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Failed to save file']);
            exit;
        }

        // Load current assets config
        $assetsConfigFile = __DIR__ . '/../config/site-assets.json';
        $assetsConfig = [];
        if (file_exists($assetsConfigFile)) {
            $assetsConfig = json_decode(file_get_contents($assetsConfigFile), true) ?: [];
        }

        // Delete old social logo if it exists
        if (!empty($assetsConfig['socialLogo'])) {
            $oldPath = ltrim($assetsConfig['socialLogo'], '/');
            $oldFull = realpath(__DIR__ . '/../' . $oldPath);
            $uploadDirReal = realpath($uploadDir);
            if ($oldFull && $uploadDirReal && strpos($oldFull, $uploadDirReal) === 0 && is_file($oldFull)) {
                @unlink($oldFull);
            }
        }

        // Update config
        $assetsConfig['socialLogo'] = $publicPrefix . $filename;
        file_put_contents($assetsConfigFile, json_encode($assetsConfig, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        echo json_encode(['success' => true, 'url' => $publicPrefix . $filename]);
        exit;
    }

    // Shared validation
    $fileKey = null;
    if (isset($_FILES['reviewImage'])) $fileKey = 'reviewImage';
    elseif (isset($_FILES['puppyImage'])) $fileKey = 'puppyImage';

    if ($fileKey) {
        $file = $_FILES[$fileKey];
        if ($file['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Upload error']);
            exit;
        }

        $maxBytes = 5 * 1024 * 1024;
        if ($file['size'] > $maxBytes) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'File too large']);
            exit;
        }

        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mime = $finfo->file($file['tmp_name']);
        $allowed = ['image/jpeg' => 'jpg', 'image/png' => 'png', 'image/webp' => 'webp'];
        if (!isset($allowed[$mime])) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Invalid file type']);
            exit;
        }

        // Choose upload directory based on file key
        if ($fileKey === 'reviewImage') {
            $uploadDir = __DIR__ . '/../uploads/reviews';
            $publicPrefix = '/uploads/reviews/';
        } else {
            $uploadDir = __DIR__ . '/../uploads/puppies';
            $publicPrefix = '/uploads/puppies/';
        }

        if (!is_dir($uploadDir)) {
            mkdir($uploadDir, 0755, true);
            // Create a safe .htaccess to prevent script execution and set caching for images
            $ht = "Options -Indexes\n\n# Deny execution of scripts in the uploads directory\n<FilesMatch \"\\.(php|php5|phtml|pl|py|jsp|asp|sh|cgi)$\">\n  Require all denied\n</FilesMatch>\n\n# Allow images and static assets to be served\n<IfModule mod_headers.c>\n  <FilesMatch \"\\.(jpg|jpeg|png|webp|gif)$\">\n    Header set Cache-Control \"max-age=604800, public\"\n  </FilesMatch>\n</IfModule>\n";
            @file_put_contents($uploadDir . '/.htaccess', $ht);
        }

        $ext = $allowed[$mime];
        $targetName = $_POST['targetName'] ?? null;

        if ($targetName && is_string($targetName) && trim($targetName) !== '') {
            $base = sanitize_base_name($targetName);
            if ($base === '') $base = bin2hex(random_bytes(8));
            $filename = $base . '.' . $ext;
            $destination = $uploadDir . '/' . $filename;
            if (file_exists($destination)) {
                $suffix = bin2hex(random_bytes(4));
                $filename = $base . '-' . $suffix . '.' . $ext;
                $destination = $uploadDir . '/' . $filename;
            }
        } else {
            $base = bin2hex(random_bytes(8));
            $filename = $base . '.' . $ext;
            $destination = $uploadDir . '/' . $filename;
        }

        if (!move_uploaded_file($file['tmp_name'], $destination)) {
            http_response_code(500);
            echo json_encode(['success' => false, 'message' => 'Failed to save file']);
            exit;
        }

        $publicPath = $publicPrefix . $filename;
        echo json_encode(['success' => true, 'url' => $publicPath]);
        exit;
    }
}

// For non-file requests, parse JSON body
$input = file_get_contents('php://input');
$data = json_decode($input, true);

if (!$data) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid JSON input']);
    exit;
}

// Helper: delete an internal image file if it resides under uploads/reviews or uploads/puppies
function try_delete_internal_image($imagePath) {
    if (empty($imagePath)) return;
    // Skip external URLs
    if (preg_match('#^https?://#i', $imagePath)) return;

    // Normalize and resolve paths
    $rel = ltrim($imagePath, '/');
    $full = realpath(__DIR__ . '/../' . $rel);
    
    // Check if file is inside uploads/reviews
    $reviewsDir = realpath(__DIR__ . '/../uploads/reviews');
    if ($full && $reviewsDir && strpos($full, $reviewsDir) === 0 && is_file($full)) {
        @unlink($full);
        return;
    }
    
    // Check if file is inside uploads/puppies
    $puppiesDir = realpath(__DIR__ . '/../uploads/puppies');
    if ($full && $puppiesDir && strpos($full, $puppiesDir) === 0 && is_file($full)) {
        @unlink($full);
        return;
    }
}

$action = $data['action'] ?? null;

// Helper to load config safely
function load_config_file() {
    $configPath = __DIR__ . '/../config/site-config.json';
    if (!file_exists($configPath)) return [$configPath, []];
    $json = file_get_contents($configPath);
    $cfg = json_decode($json, true);
    return [$configPath, $cfg ?: []];
}

// Default fallback credentials (legacy fallback)
define('DEFAULT_ADMIN_USERNAME', 'admin');
define('DEFAULT_ADMIN_PASSWORD', 'admin');

// Handle admin_login action (authenticate credentials server-side)
if ($action === 'admin_login') {
    $username = $data['username'] ?? '';
    $password = $data['password'] ?? '';

    list($configPath, $cfg) = load_config_file();

    // Use config username if available, otherwise fallback to default
    $cfgUser = $cfg['adminUsername'] ?? DEFAULT_ADMIN_USERNAME;

    // If a password hash exists, verify against it
    if (!empty($cfg['adminPasswordHash'])) {
        $hash = $cfg['adminPasswordHash'];
        if ($username === $cfgUser && password_verify($password, $hash)) {
            // mark session authenticated
            session_regenerate_id(true);
            $_SESSION['admin_logged_in'] = true;
            $_SESSION['admin_user'] = $cfgUser;
            echo json_encode(['success' => true]);
            exit;
        } else {
            echo json_encode(['success' => false, 'message' => 'Invalid credentials']);
            exit;
        }
    }

    // If legacy plaintext password exists in config, compare and migrate to hash
    if (!empty($cfg['adminPassword'])) {
        if ($username === $cfgUser && $password === $cfg['adminPassword']) {
            // Migrate: create a hash and save to config (remove plaintext)
            $cfg['adminPasswordHash'] = password_hash($cfg['adminPassword'], PASSWORD_DEFAULT);
            unset($cfg['adminPassword']);
            if ($configPath) {
                file_put_contents($configPath, json_encode($cfg, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
            }
            // mark session authenticated
            session_regenerate_id(true);
            $_SESSION['admin_logged_in'] = true;
            $_SESSION['admin_user'] = $cfgUser;
            echo json_encode(['success' => true, 'migrated' => true]);
            exit;
        } else {
            echo json_encode(['success' => false, 'message' => 'Invalid credentials']);
            exit;
        }
    }

    // No password in config — fallback to defaults for backwards compatibility
    if ($username === ($cfg['adminUsername'] ?? DEFAULT_ADMIN_USERNAME) && $password === DEFAULT_ADMIN_PASSWORD) {
        // mark session authenticated for default fallback
        session_regenerate_id(true);
        $_SESSION['admin_logged_in'] = true;
        $_SESSION['admin_user'] = $cfg['adminUsername'] ?? DEFAULT_ADMIN_USERNAME;
        echo json_encode(['success' => true, 'message' => 'Authenticated with default credentials']);
        exit;
    }

    echo json_encode(['success' => false, 'message' => 'Invalid credentials']);
    exit;
}

// Handle save_config action
if ($action === 'save_config') {
    $config = $data['config'] ?? null;
    
    if (!$config || !is_array($config)) {
        echo json_encode(['success' => false, 'message' => 'No config data provided']);
        exit;
    }
    
    // Path to config file (in config subfolder)
    $configPath = __DIR__ . '/../config/site-config.json';
    
    // Validate that the config file exists
    if (!file_exists($configPath)) {
        echo json_encode(['success' => false, 'message' => 'Config file not found']);
        exit;
    }
    
    // Prepare the data
    // Coerce known numeric/boolean config keys to proper types before saving
    $numericKeys = ['confirmationDelayMs'];
    foreach ($numericKeys as $k) {
        if (array_key_exists($k, $config)) {
            // Convert empty strings to 0 and numeric strings to int
            if ($config[$k] === '' || $config[$k] === null) {
                $config[$k] = 0;
            } else {
                $config[$k] = intval($config[$k]);
            }
        }
    }

    // If adminPassword was provided, hash it and store as adminPasswordHash
    if (array_key_exists('adminPassword', $config) && $config['adminPassword'] !== '') {
        // Create a bcrypt hash
        $hash = password_hash($config['adminPassword'], PASSWORD_DEFAULT);
        if ($hash !== false) {
            $config['adminPasswordHash'] = $hash;
        }
        // Remove plaintext password before saving
        unset($config['adminPassword']);
    }

    $jsonData = json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    
    // Write the file
    if (file_put_contents($configPath, $jsonData) !== false) {
        echo json_encode(['success' => true, 'message' => 'Configuration saved successfully']);
        exit;
    } else {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to write config file']);
        exit;
    }
}

// Handle save_smtp_config action
if ($action === 'save_smtp_config') {
    $smtp = $data['smtp'] ?? null;
    if (!$smtp || !is_array($smtp)) {
        echo json_encode(['success' => false, 'message' => 'No SMTP config provided']);
        exit;
    }

    $smtpPath = __DIR__ . '/../config/smtp-config.json';

    // Build sanitized object
    // Load existing config so we can preserve password when not provided
    $existing = [];
    if (file_exists($smtpPath)) {
        $existingJson = file_get_contents($smtpPath);
        $existing = json_decode($existingJson, true) ?: [];
    }

    $out = [];
    $out['host'] = isset($smtp['host']) ? trim($smtp['host']) : ($existing['host'] ?? '');
    $out['username'] = isset($smtp['username']) ? trim($smtp['username']) : ($existing['username'] ?? '');
    // Only overwrite password if provided; otherwise preserve existing password
    if (isset($smtp['password']) && $smtp['password'] !== '') {
        $out['password'] = $smtp['password'];
    } else if (isset($existing['password'])) {
        $out['password'] = $existing['password'];
    }
    $out['encryption'] = isset($smtp['encryption']) ? trim($smtp['encryption']) : ($existing['encryption'] ?? 'smtps');
    $out['port'] = isset($smtp['port']) ? intval($smtp['port']) : ($existing['port'] ?? 465);
    // timeout and explicit fromEmail removed from UI; preserve existing values if present
    if (isset($existing['timeout'])) $out['timeout'] = intval($existing['timeout']);
    if (isset($existing['fromEmail'])) $out['fromEmail'] = $existing['fromEmail'];

    $jsonData = json_encode($out, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    if (file_put_contents($smtpPath, $jsonData) !== false) {
        echo json_encode(['success' => true, 'message' => 'SMTP config saved']);
        exit;
    } else {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to write SMTP config']);
        exit;
    }
}

// Check session action
if ($action === 'check_session') {
    if (!empty($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true) {
        echo json_encode(['success' => true, 'user' => $_SESSION['admin_user'] ?? null]);
    } else {
        echo json_encode(['success' => false]);
    }
    exit;
}

// Logout action
if ($action === 'admin_logout') {
    // Clear session
    $_SESSION = [];
    if (ini_get('session.use_cookies')) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params['path'], $params['domain'], $params['secure'], $params['httponly']
        );
    }
    session_destroy();
    echo json_encode(['success' => true]);
    exit;
}

// Change credentials action (requires current password)
if ($action === 'change_admin_credentials') {
    $current = $data['currentPassword'] ?? '';
    $newPass = $data['newPassword'] ?? '';
    $newUser = $data['newUsername'] ?? null;

    // Require current password and at least one of new password or new username
    if ($current === '' || ($newPass === '' && ($newUser === null || $newUser === ''))) {
        echo json_encode(['success' => false, 'message' => 'Current password and a new username or new password are required']);
        exit;
    }

    list($configPath, $cfg) = load_config_file();

    // Determine current username
    $cfgUser = $cfg['adminUsername'] ?? DEFAULT_ADMIN_USERNAME;

    $verified = false;

    // Verify against hash
    if (!empty($cfg['adminPasswordHash'])) {
        if (password_verify($current, $cfg['adminPasswordHash']) && ($data['username'] ?? $cfgUser) === $cfgUser) {
            $verified = true;
        }
    }

    // Legacy plaintext verify
    if (!$verified && !empty($cfg['adminPassword'])) {
        if ($current === $cfg['adminPassword'] && ($data['username'] ?? $cfgUser) === $cfgUser) {
            $verified = true;
        }
    }

    // Default fallback (only if config has no password entries)
    if (!$verified && empty($cfg['adminPasswordHash']) && empty($cfg['adminPassword'])) {
        if ($current === DEFAULT_ADMIN_PASSWORD && ($data['username'] ?? $cfgUser) === $cfgUser) {
            $verified = true;
        }
    }

    if (!$verified) {
        echo json_encode(['success' => false, 'message' => 'Current password is incorrect']);
        exit;
    }

    // Update config: set new username if provided, set new password hash only if a new password was provided
    if ($newUser !== null && $newUser !== '') {
        $cfg['adminUsername'] = $newUser;
    }

    if ($newPass !== '') {
        $cfg['adminPasswordHash'] = password_hash($newPass, PASSWORD_DEFAULT);
    }

    if (isset($cfg['adminPassword'])) unset($cfg['adminPassword']);

    if ($configPath) {
        if (file_put_contents($configPath, json_encode($cfg, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)) === false) {
            echo json_encode(['success' => false, 'message' => 'Failed to write config file']);
            exit;
        }
    }

    // Refresh session user
    session_regenerate_id(true);
    $_SESSION['admin_logged_in'] = true;
    $_SESSION['admin_user'] = $cfg['adminUsername'] ?? $cfgUser;

    echo json_encode(['success' => true, 'message' => 'Credentials updated']);
    exit;
}

// Handle delete_review action
if ($action === 'delete_review') {
    $reviewId = $data['review_id'] ?? null;
    
    if ($reviewId === null) {
        echo json_encode(['success' => false, 'message' => 'No review ID provided']);
        exit;
    }
    
    // Path to reviews file
    $reviewsPath = __DIR__ . '/../reviews/reviews.json';
    
    // Validate that the reviews file exists
    if (!file_exists($reviewsPath)) {
        echo json_encode(['success' => false, 'message' => 'Reviews file not found']);
        exit;
    }
    
    // Read the reviews
    $reviewsJson = file_get_contents($reviewsPath);
    $reviews = json_decode($reviewsJson, true);
    
    if (!is_array($reviews)) {
        echo json_encode(['success' => false, 'message' => 'Invalid reviews data']);
        exit;
    }
    
    // Before removing, try to delete internal image for the review being removed
    foreach ($reviews as $r) {
        if (isset($r['id']) && $r['id'] == $reviewId) {
            if (!empty($r['image'])) {
                try_delete_internal_image($r['image']);
            }
            break;
        }
    }

    // Filter out the review with the matching ID
    $updatedReviews = array_filter($reviews, function($review) use ($reviewId) {
        return $review['id'] != $reviewId;
    });
    
    // Check if a review was actually deleted
    if (count($updatedReviews) === count($reviews)) {
        echo json_encode(['success' => false, 'message' => 'Review not found']);
        exit;
    }
    
    // Re-index the array to remove array gaps
    $updatedReviews = array_values($updatedReviews);
    
    // Write the updated reviews back to file
    $jsonData = json_encode($updatedReviews, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    
    if (file_put_contents($reviewsPath, $jsonData) !== false) {
        echo json_encode(['success' => true, 'message' => 'Review deleted successfully']);
        exit;
    } else {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to delete review']);
        exit;
    }
}

// Handle add_review action
if ($action === 'add_review') {
    $review = $data['review'] ?? null;
    
    if (!$review || !is_array($review)) {
        echo json_encode(['success' => false, 'message' => 'No review data provided']);
        exit;
    }
    
    // Validate required fields
    $requiredFields = ['id', 'title', 'body', 'author', 'ownerOf', 'rating'];
    foreach ($requiredFields as $field) {
        if (!isset($review[$field]) || $review[$field] === '') {
            echo json_encode(['success' => false, 'message' => "Missing required field: $field"]);
            exit;
        }
    }
    
    // Path to reviews file
    $reviewsPath = __DIR__ . '/../reviews/reviews.json';
    
    // Validate that the reviews file exists
    if (!file_exists($reviewsPath)) {
        echo json_encode(['success' => false, 'message' => 'Reviews file not found']);
        exit;
    }
    
    // Read the existing reviews
    $reviewsJson = file_get_contents($reviewsPath);
    $reviews = json_decode($reviewsJson, true);
    
    if (!is_array($reviews)) {
        echo json_encode(['success' => false, 'message' => 'Invalid reviews data']);
        exit;
    }
    
    // Check if ID already exists
    $idExists = array_filter($reviews, function($r) use ($review) {
        return $r['id'] == $review['id'];
    });
    
    if (!empty($idExists)) {
        echo json_encode(['success' => false, 'message' => 'Review ID already exists']);
        exit;
    }
    
    // Add the new review
    // Ensure imageAlt is present — default to the review title for accessibility
    if (!isset($review['imageAlt']) || $review['imageAlt'] === '') {
        $review['imageAlt'] = isset($review['title']) && $review['title'] !== '' ? $review['title'] : 'New Puppy Owner';
    }

    $reviews[] = $review;
    
    // Write the updated reviews back to file
    $jsonData = json_encode($reviews, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    
    if (file_put_contents($reviewsPath, $jsonData) !== false) {
        echo json_encode(['success' => true, 'message' => 'Review added successfully']);
        exit;
    } else {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to add review']);
        exit;
    }
}

// Handle edit_review action
if ($action === 'edit_review') {
    $review = $data['review'] ?? null;
    if (!$review || !is_array($review)) {
        echo json_encode(['success' => false, 'message' => 'No review data provided']);
        exit;
    }

    // Validate required fields exist
    if (!isset($review['id'])) {
        echo json_encode(['success' => false, 'message' => 'Review ID is required']);
        exit;
    }

    $reviewsPath = __DIR__ . '/../reviews/reviews.json';
    if (!file_exists($reviewsPath)) {
        echo json_encode(['success' => false, 'message' => 'Reviews file not found']);
        exit;
    }

    $reviewsJson = file_get_contents($reviewsPath);
    $reviewsArr = json_decode($reviewsJson, true);
    if (!is_array($reviewsArr)) {
        echo json_encode(['success' => false, 'message' => 'Invalid reviews data']);
        exit;
    }

    $found = false;
    foreach ($reviewsArr as $k => $r) {
        if (isset($r['id']) && $r['id'] == $review['id']) {
            // Keep old image so we can delete it if replaced
            $oldImage = isset($reviewsArr[$k]['image']) ? $reviewsArr[$k]['image'] : null;

            // merge fields — only allow known keys
            $allowed = ['title','body','author','ownerOf','rating','image','imageAlt'];
            foreach ($allowed as $key) {
                if (array_key_exists($key, $review)) {
                    $reviewsArr[$k][$key] = $review[$key];
                }
            }

            // If image was changed and old image was internal, delete old file
            if (array_key_exists('image', $review)) {
                $newImage = $review['image'];
                if ($oldImage && $oldImage !== $newImage) {
                    try_delete_internal_image($oldImage);
                }
            }

            $found = true;
            break;
        }
    }

    if (!$found) {
        echo json_encode(['success' => false, 'message' => 'Review not found']);
        exit;
    }

    // Write back to file
    $outJson = json_encode($reviewsArr, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    if (file_put_contents($reviewsPath, $outJson) !== false) {
        echo json_encode(['success' => true, 'message' => 'Review updated']);
        exit;
    } else {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to save review']);
        exit;
    }
}

// Unknown action
// ------------------ Puppies CRUD ------------------
if ($action === 'list_puppies') {
    $puppiesPath = __DIR__ . '/../data/puppies.json';
    if (!file_exists($puppiesPath)) {
        echo json_encode(['success' => true, 'puppies' => []]);
        exit;
    }
    $json = file_get_contents($puppiesPath);
    $obj = json_decode($json, true);
    $list = [];
    if (is_array($obj) && isset($obj['puppies']) && is_array($obj['puppies'])) $list = $obj['puppies'];
    echo json_encode(['success' => true, 'puppies' => $list]);
    exit;
}

if ($action === 'add_puppy') {
    $puppy = $data['puppy'] ?? null;
    if (!$puppy || !is_array($puppy)) {
        echo json_encode(['success' => false, 'message' => 'No puppy data provided']);
        exit;
    }

    $puppiesPath = __DIR__ . '/../data/puppies.json';
    $obj = ['puppies' => []];
    if (file_exists($puppiesPath)) {
        $json = file_get_contents($puppiesPath);
        $obj = json_decode($json, true) ?: ['puppies' => []];
    }

    $arr = $obj['puppies'];
    // determine id
    $nextId = 1;
    if (!empty($arr)) {
        $ids = array_map(function($p){ return isset($p['id']) ? intval($p['id']) : 0; }, $arr);
        $nextId = max($ids) + 1;
    }

    if (!isset($puppy['id']) || !$puppy['id']) {
        $puppy['id'] = $nextId;
    } else {
        // ensure id is unique
        foreach ($arr as $existing) {
            if (isset($existing['id']) && $existing['id'] == $puppy['id']) {
                echo json_encode(['success' => false, 'message' => 'Puppy ID already exists']);
                exit;
            }
        }
    }

    // Normalize arrays (images only — badges are hard-coded and not stored in JSON)
    if (!isset($puppy['images']) || !is_array($puppy['images'])) $puppy['images'] = isset($puppy['images']) && $puppy['images'] !== '' ? array_map('trim', explode(',', $puppy['images'])) : [];

    $arr[] = $puppy;
    $obj['puppies'] = $arr;

    if (file_put_contents($puppiesPath, json_encode($obj, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)) !== false) {
        echo json_encode(['success' => true, 'puppy' => $puppy]);
        exit;
    } else {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to write puppies file']);
        exit;
    }
}

if ($action === 'edit_puppy') {
    $puppy = $data['puppy'] ?? null;
    if (!$puppy || !is_array($puppy) || !isset($puppy['id'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid puppy data']);
        exit;
    }

    $puppiesPath = __DIR__ . '/../data/puppies.json';
    if (!file_exists($puppiesPath)) {
        echo json_encode(['success' => false, 'message' => 'Puppies file not found']);
        exit;
    }
    $json = file_get_contents($puppiesPath);
    $obj = json_decode($json, true);
    if (!is_array($obj) || !isset($obj['puppies']) || !is_array($obj['puppies'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid puppies data']);
        exit;
    }

    $found = false;
    foreach ($obj['puppies'] as $k => $p) {
        if (isset($p['id']) && $p['id'] == $puppy['id']) {
            // Track old images to possibly delete internal files
            $oldImages = isset($obj['puppies'][$k]['images']) ? $obj['puppies'][$k]['images'] : [];

            // Only allow known keys to be updated (badges are hard-coded elsewhere)
            $allowed = ['name','slug','age','sex','status','price','images'];
            foreach ($allowed as $key) {
                if (array_key_exists($key, $puppy)) {
                    $obj['puppies'][$k][$key] = $puppy[$key];
                }
            }

            // If images changed, delete old internal images that are no longer present
            if (array_key_exists('images', $puppy)) {
                $newImages = is_array($puppy['images']) ? $puppy['images'] : [];
                foreach ($oldImages as $img) {
                    if ($img && !in_array($img, $newImages)) {
                        try_delete_internal_image($img);
                    }
                }
            }

            $found = true;
            break;
        }
    }

    if (!$found) {
        echo json_encode(['success' => false, 'message' => 'Puppy not found']);
        exit;
    }

    if (file_put_contents($puppiesPath, json_encode($obj, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)) !== false) {
        echo json_encode(['success' => true, 'message' => 'Puppy updated']);
        exit;
    } else {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to save puppies']);
        exit;
    }
}

if ($action === 'delete_puppy') {
    $puppyId = $data['puppy_id'] ?? null;
    if ($puppyId === null) {
        echo json_encode(['success' => false, 'message' => 'No puppy ID provided']);
        exit;
    }

    $puppiesPath = __DIR__ . '/../data/puppies.json';
    if (!file_exists($puppiesPath)) {
        echo json_encode(['success' => false, 'message' => 'Puppies file not found']);
        exit;
    }

    $json = file_get_contents($puppiesPath);
    $obj = json_decode($json, true);
    if (!is_array($obj) || !isset($obj['puppies']) || !is_array($obj['puppies'])) {
        echo json_encode(['success' => false, 'message' => 'Invalid puppies data']);
        exit;
    }

    $deleted = false;
    $new = [];
    foreach ($obj['puppies'] as $p) {
        if (isset($p['id']) && $p['id'] == $puppyId) {
            // Delete internal images
            if (!empty($p['images']) && is_array($p['images'])) {
                foreach ($p['images'] as $img) {
                    try_delete_internal_image($img);
                }
            }
            $deleted = true;
            continue;
        }
        $new[] = $p;
    }

    if (!$deleted) {
        echo json_encode(['success' => false, 'message' => 'Puppy not found']);
        exit;
    }

    $obj['puppies'] = array_values($new);
    if (file_put_contents($puppiesPath, json_encode($obj, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES)) !== false) {
        echo json_encode(['success' => true, 'message' => 'Puppy deleted']);
        exit;
    } else {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to delete puppy']);
        exit;
    }
}

if ($action === 'deleteHomeImage') {
    $position = $data['position'] ?? null;

    if (!in_array($position, ['image1', 'image2'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid position']);
        exit;
    }

    // Load images config
    $imagesConfigFile = __DIR__ . '/../config/home-images.json';
    $imagesConfig = [];
    if (file_exists($imagesConfigFile)) {
        $imagesConfig = json_decode(file_get_contents($imagesConfigFile), true) ?: [];
    }

    // Delete old image file if it exists
    if (!empty($imagesConfig[$position])) {
        $oldPath = ltrim($imagesConfig[$position], '/');
        $oldFull = realpath(__DIR__ . '/../' . $oldPath);
        $uploadDir = realpath(__DIR__ . '/../uploads/images');
        if ($oldFull && $uploadDir && strpos($oldFull, $uploadDir) === 0 && is_file($oldFull)) {
            @unlink($oldFull);
        }
        
        // Remove from config
        unset($imagesConfig[$position]);
        file_put_contents($imagesConfigFile, json_encode($imagesConfig, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));

        // Update homepage HTML to remove hardcoded URL
        updateHomepageImageUrls();
    }

    echo json_encode(['success' => true, 'message' => 'Image deleted']);
    exit;
}

if ($action === 'deleteSiteAsset') {
    $assetType = $data['assetType'] ?? null;

    if (!in_array($assetType, ['favicon', 'socialLogo'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid asset type']);
        exit;
    }

    // Load assets config
    $assetsConfigFile = __DIR__ . '/../config/site-assets.json';
    $assetsConfig = [];
    if (file_exists($assetsConfigFile)) {
        $assetsConfig = json_decode(file_get_contents($assetsConfigFile), true) ?: [];
    }

    // Delete asset file if it exists
    if (!empty($assetsConfig[$assetType])) {
        $oldPath = ltrim($assetsConfig[$assetType], '/');
        $oldFull = realpath(__DIR__ . '/../' . $oldPath);
        $uploadDir = realpath(__DIR__ . '/../uploads/images');
        if ($oldFull && $uploadDir && strpos($oldFull, $uploadDir) === 0 && is_file($oldFull)) {
            @unlink($oldFull);
        }
        
        // Remove from config
        unset($assetsConfig[$assetType]);
        file_put_contents($assetsConfigFile, json_encode($assetsConfig, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
    }

    echo json_encode(['success' => true, 'message' => 'Asset deleted']);
    exit;
}

echo json_encode(['success' => false, 'message' => 'Unknown action']);
exit;
?>
